/**
* \file: common.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Rexaline Xavier  /  RexalineInfancia.Xavier@in.bosch.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include "common.h"
#include "encryption/signature.h"

const char *error_code_strings[] = {
		"Everything fine",
		"API calls have been used in an invalid way",
		"Resource issues (memory)",
		"Unable to access the persistent state file (write access)",
		"A change to an invalid level has been requested",
		"Scripts have been modified",
		"The execution of one or more scripts failed",
		"Execution of one or more scripts timed out",
		"The level configuration was modified",
		"The key to verify the signature has been modified",
		"The signature on the response is invalid",
		"The requested challenge timed out",
		"The challenge as part of the response is invalid",
		"An level change is ongoing",
		"An level recovery is ongoing",
		"The script execution timed out",
		"The recovery on script exec tmout failed,waiting for a recovery on startup to heal",
		"Signature verification for data failed ",
		"Invalid arguments are passed while starting ALD",
		"Printed Help display",
		"The API had some problems connecting to the dbus system bus",
		"Wrapped state file and signature DB pub key got corrupted",
		"The configuration of the requested level is invalid",
		"An invalid signature db has been detected",
		"The signature db file could not be created",
		"File not found",
		"An error occurred during private/public key generation",
		"Unable to call openssl to generate new private/public key pairs",
		"A private or public key read in from file is invalid",
		"The signing of a file failed",
		"The signature that is tried to be read out from file is invalid",
		"The en/decryption failed",
		"Invalid type of the file",
		"Mandatory file not found,this will result is failure of function",
		"Stat for the file has returned error",
		"Invalid or unknown command"
};

error_code_t common_init(void)
{
#if !GLIB_CHECK_VERSION (2, 35, 0)
    g_type_init ();
#endif

    signature_init();

    return RESULT_OK;
}

void common_deinit(void)
{
    signature_deinit();
}

error_code_t common_connect_ALD(Change_level **change_level_inface)
{
	GError *err = NULL;

	*change_level_inface = change_level_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM,
			G_DBUS_PROXY_FLAGS_NONE,"com.adit.de.ALD", "/com/adit/de/ALD/change_level",NULL, &err);

	if (err != NULL)
	{
		fprintf(stderr, "Unable to connect the ALD: %s\n", err->message);
		g_error_free(err);
		return RESULT_DAEMON_BUSY_LEVEL_CHANGE;
	}

	return RESULT_OK;
}

void common_disconnect_ald(Change_level **change_level_iface)
{
	if(change_level_iface != NULL)
		g_object_unref(*change_level_iface);
}

void common_initialize_response(challenge_response_t *response, security_level_t target_level, const char *serial)
{
    memset(response, 0, sizeof(challenge_response_t));

    response->targeted_level = target_level;

    if (serial != NULL) {
        strncpy((char *)&(response->serial_number), serial, RESPONSE_SERIAL_NUMBER_SIZE);

        if (strlen(serial) > RESPONSE_SERIAL_NUMBER_SIZE) {
            fprintf(stderr, "Warning: Serial number concatenated to first %u characters (i.e. %.*s)\n",
                    RESPONSE_SERIAL_NUMBER_SIZE, RESPONSE_SERIAL_NUMBER_SIZE, serial);
        }
    }
}

error_code_t common_get_challenge_and_add_to_response(Change_level *change_level_iface,challenge_response_t *response,const char *ecu_id)
{
	const challenge_t *challenge;
	GVariant *challenge_ser;
	GError *err = NULL;

	if (strlen(ecu_id) > CHALLENGE_ECU_ID_SIZE)  {
        fprintf(stderr, "Warning: EcuID concatenated to first %u characters (i.e. %.*s)\n",
                CHALLENGE_ECU_ID_SIZE, CHALLENGE_ECU_ID_SIZE, ecu_id);
	}

	if (!change_level_call_challenge_sync(change_level_iface,ecu_id,&challenge_ser,NULL,&err))
	{
		fprintf(stderr, "Unable to receive a challenge from the ALD: %s\n", err->message);
		g_error_free(err);
		return RESULT_INVALID;
	}
	challenge = g_variant_get_data(challenge_ser);
	printf("--------------------- Challenge Received -----------------------------------------------------\n");
	printf("Time:\t\t%f secs\n", challenge->tv_sec+(challenge->tv_usec/1000000.0));
	printf("ALD protocol version:%d.%d\n",challenge->ALD_protocol_major,challenge->ALD_protocol_minor);
	printf("----------------------------------------------------------------------------------------------\n");

	//Fill the response buffer
	memcpy(&response->challenge,challenge,sizeof(challenge_t));
	response->client_ALD_protocol_major = CLIENT_ALD_PROTOCOL_MAJOR_VERSION;
	response->client_ALD_protocol_minor = CLIENT_ALD_PROTOCOL_MINOR_VERSION;

	printf("ALD client protocol version:%d.%d\n",response->client_ALD_protocol_major,response->client_ALD_protocol_minor);

	g_variant_unref(challenge_ser);
	return RESULT_OK;
}

error_code_t common_calculate_signature(const char *levelprivkey_path,challenge_response_t *response)
{
	error_code_t result;
	EVP_PKEY *privkey = NULL;
	char signature[RESPONSE_SIGNATURE_SIZE_USED] = {0};

	result = signature_read_private_key(levelprivkey_path,&privkey);

	if (result == RESULT_OK)
		result = signature_create_for_data_block(response, sizeof(challenge_response_t)-RESPONSE_SIGNATURE_SIZE_MAX,signature,privkey);

	if (result == RESULT_OK)
		memcpy(response->signature,signature,RESPONSE_SIGNATURE_SIZE_USED);

	return result;
}

error_code_t common_send_response(Change_level *change_level_iface,challenge_response_t *response)
{
	GError *err = NULL;
	GVariant *response_ser;
	error_code_t result;

	response_ser = g_variant_new_from_data((const GVariantType *)"ay",response,sizeof(challenge_response_t),
			TRUE,NULL, NULL);

	if (!change_level_call_response_sync(change_level_iface,response_ser,(guint *)&result,NULL,&err))
	{
		fprintf(stderr, "Level change request failed: %s\n", err->message);
		g_error_free(err);
		return RESULT_INVALID;
	}

	if (result != RESULT_OK)
		fprintf(stderr, "Level change request failed(Code: %d): %s\n", result, error_code_strings[result]);

	return result;
}

error_code_t common_lock_device(Change_level *change_level_iface)
{
    error_code_t result = RESULT_OK;
    GError *err = NULL;

    if (!change_level_call_lock_device_sync(change_level_iface,(guint *)&result,NULL,&err))
    {
        fprintf(stderr, "Unable to lock the device: %s\n", err->message);
        g_error_free(err);
        result = RESULT_INVALID;
    }

    return result;

}


